// Copyright 2021 Axiom-Team
//
// This file is part of Duniter-v2S.
//
// Duniter-v2S is free software: you can redistribute it and/or modify
// it under the terms of the GNU Affero General Public License as published by
// the Free Software Foundation, version 3 of the License.
//
// Duniter-v2S is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU Affero General Public License for more details.
//
// You should have received a copy of the GNU Affero General Public License
// along with Duniter-v2S. If not, see <https://www.gnu.org/licenses/>.

#[macro_export]
macro_rules! pallets_config {
    () => {
        // SYSTEM //

        parameter_types! {
            pub const Version: RuntimeVersion = VERSION;
        }

        impl frame_system::Config for Runtime {
            /// The data to be stored in an account.
            type AccountData = pallet_duniter_account::AccountData<Balance, IdtyIndex>;
            /// The identifier used to distinguish between accounts.
            type AccountId = AccountId;
            /// The basic call filter to use in dispatchable.
            type BaseCallFilter = BaseCallFilter;
            /// The block type for the runtime.
            type Block = Block;
            /// Maximum number of block number to block hash mappings to keep (oldest pruned first).
            type BlockHashCount = BlockHashCount;
            /// The maximum length of a block (in bytes).
            type BlockLength = BlockLength;
            /// Block & extrinsics weights: base values and limits.
            type BlockWeights = BlockWeights;
            /// The weight of database operations that the runtime can invoke.
            type DbWeight = DbWeight;
            /// The weight of transaction extensions.
            type ExtensionsWeightInfo = weights::frame_system_extensions::WeightInfo<Runtime>;
            /// The type for hashing blocks and tries.
            type Hash = Hash;
            /// The hashing algorithm used.
            type Hashing = BlakeTwo256;
            /// The lookup mechanism to get account ID from whatever is passed in dispatchers.
            type Lookup = AccountIdLookup<AccountId, ()>;
            type MaxConsumers = frame_support::traits::ConstU32<16>;
            type MultiBlockMigrator = ();
            /// The type for storing how many extrinsics an account has signed.
            type Nonce = node_primitives::Nonce;
            /// What to do if an account is fully reaped from the system.
            type OnKilledAccount = ();
            /// What to do if a new account is created.
            type OnNewAccount = ();
            /// The set code logic, just the default since we're not a parachain.
            type OnSetCode = ();
            /// Converts a module to the index of the module in `construct_runtime!`.
            ///
            /// This type is being generated by `construct_runtime!`.
            type PalletInfo = PalletInfo;
            type PostInherents = ();
            type PostTransactions = ();
            type PreInherents = ();
            /// The aggregated dispatch type that is available for extrinsics.
            type RuntimeCall = RuntimeCall;
            /// The ubiquitous event type.
            type RuntimeEvent = RuntimeEvent;
            /// The ubiquitous origin type.
            type RuntimeOrigin = RuntimeOrigin;
            type RuntimeTask = ();
            /// This is used as an identifier of the chain. 42 is the generic substrate prefix.
            type SS58Prefix = SS58Prefix;
            type SingleBlockMigrations = ();
            /// Weight information for the extrinsics of this pallet.
            type SystemWeightInfo = weights::frame_system::WeightInfo<Runtime>;
            /// Version of the runtime.
            type Version = Version;
        }

        // SCHEDULER //

        parameter_types! {
            pub MaximumSchedulerWeight: Weight = Perbill::from_percent(80) *
                BlockWeights::get().max_block;
            pub const MaxScheduledPerBlock: u32 = 50;
            pub const NoPreimagePostponement: Option<u32> = Some(10);
        }
        impl pallet_scheduler::Config for Runtime {
            type MaxScheduledPerBlock = MaxScheduledPerBlock;
            type MaximumWeight = MaximumSchedulerWeight;
            type OriginPrivilegeCmp = EqualPrivilegeOnly;
            type PalletsOrigin = OriginCaller;
            type Preimages = Preimage;
            type RuntimeCall = RuntimeCall;
            type RuntimeEvent = RuntimeEvent;
            type RuntimeOrigin = RuntimeOrigin;
            type ScheduleOrigin = EnsureRoot<AccountId>;
            type WeightInfo = weights::pallet_scheduler::WeightInfo<Runtime>;
        }

        // ACCOUNT //

        impl pallet_duniter_account::Config for Runtime {
            // does currency adapter in any case, but adds "refund with quota" feature
            type InnerOnChargeTransaction =
                FungibleAdapter<Balances, HandleFees<TreasuryAccount, Balances>>;
            type Refund = Quota;
            type RuntimeEvent = RuntimeEvent;
            type WeightInfo = weights::pallet_duniter_account::WeightInfo<Runtime>;
        }

        parameter_types! {
            pub const ReloadRate: BlockNumber = 1 * HOURS; // faster than DAYS
            pub const MaxQuota: Balance = 1000; // 10 ĞD
            pub const MaxNominators: u32 = 64;
            pub TreasuryAccount: AccountId = Treasury::account_id();
        }
        impl pallet_quota::Config for Runtime {
            type MaxQuota = MaxQuota;
            type RefundAccount = TreasuryAccount;
            type ReloadRate = ReloadRate;
            type RuntimeEvent = RuntimeEvent;
            type WeightInfo = weights::pallet_quota::WeightInfo<Runtime>;
        }

        // BLOCK CREATION //

        impl pallet_babe::Config for Runtime {
            type DisabledValidators = Session;
            // session module is the trigger
            type EpochChangeTrigger = pallet_babe::ExternalTrigger;
            type EpochDuration = EpochDuration;
            type EquivocationReportSystem =
                pallet_babe::EquivocationReportSystem<Self, Offences, Historical, ReportLongevity>;
            type ExpectedBlockTime = ExpectedBlockTime;
            type KeyOwnerProof =
                <Historical as KeyOwnerProofSystem<(KeyTypeId, pallet_babe::AuthorityId)>>::Proof;
            type MaxAuthorities = MaxAuthorities;
            type MaxNominators = MaxNominators;
            type WeightInfo = weights::pallet_babe::WeightInfo<Runtime>;
        }

        impl pallet_timestamp::Config for Runtime {
            type MinimumPeriod = MinimumPeriod;
            type Moment = u64;
            type OnTimestampSet = (Babe, UniversalDividend);
            type WeightInfo = weights::pallet_timestamp::WeightInfo<Runtime>;
        }

        // MONEY MANAGEMENT //

        impl pallet_balances::Config for Runtime {
            type AccountStore = Account;
            type Balance = Balance;
            type DoneSlashHandler = ();
            type DustRemoval = HandleFees<TreasuryAccount, Balances>;
            type ExistentialDeposit = ExistentialDeposit;
            type FreezeIdentifier = ();
            type MaxFreezes = frame_support::pallet_prelude::ConstU32<0>;
            type MaxLocks = MaxLocks;
            type MaxReserves = frame_support::pallet_prelude::ConstU32<5>;
            type ReserveIdentifier = [u8; 8];
            type RuntimeEvent = RuntimeEvent;
            type RuntimeFreezeReason = ();
            type RuntimeHoldReason = RuntimeHoldReason;
            type WeightInfo = weights::pallet_balances::WeightInfo<Runtime>;
        }

        parameter_types! {
        pub Target: Perquintill = Perquintill::from_percent(25);
        pub MaxMultiplier: sp_runtime::FixedU128 = 10.into();
        }
        impl pallet_transaction_payment::Config for Runtime {
            type FeeMultiplierUpdate =
                common_runtime::fees::FeeMultiplier<Self, Target, MaxMultiplier>;
            type LengthToFee = common_runtime::fees::LengthToFeeImpl<Balance, Self, Target>;
            // does a filter on the call
            type OnChargeTransaction = OneshotAccount;
            type OperationalFeeMultiplier = frame_support::traits::ConstU8<5>;
            type RuntimeEvent = RuntimeEvent;
            type WeightInfo = weights::pallet_transaction_payment::WeightInfo<Runtime>;
            type WeightToFee = common_runtime::fees::WeightToFeeImpl<Balance, Self, Target>;
        }
        impl pallet_oneshot_account::Config for Runtime {
            type Currency = Balances;
            // when call is not oneshot account, fall back to duniter-account implementation
            type InnerOnChargeTransaction = Account;
            type RuntimeEvent = RuntimeEvent;
            type WeightInfo = weights::pallet_oneshot_account::WeightInfo<Runtime>;
        }

        // CONSENSUS  //

        impl pallet_authority_discovery::Config for Runtime {
            type MaxAuthorities = MaxAuthorities;
        }
        impl pallet_authority_members::Config for Runtime {
            type IsMember = SmithMembers;
            type MaxAuthorities = MaxAuthorities;
            type MemberId = IdtyIndex;
            type MemberIdOf = common_runtime::providers::IdentityIndexOf<Self>;
            type OnIncomingMember = SmithMembers;
            type OnNewSession = OnNewSessionHandler<Runtime>;
            type OnOutgoingMember = SmithMembers;
            type RemoveMemberOrigin = EnsureRoot<Self::AccountId>;
            type RuntimeEvent = RuntimeEvent;
            type WeightInfo = weights::pallet_authority_members::WeightInfo<Runtime>;
        }
        impl pallet_authorship::Config for Runtime {
            type EventHandler = ImOnline;
            type FindAuthor = pallet_session::FindAccountFromAuthorIndex<Self, Babe>;
        }
        impl pallet_im_online::Config for Runtime {
            type AuthorityId = ImOnlineId;
            #[cfg(not(feature = "runtime-benchmarks"))]
            type MaxKeys = MaxAuthorities;
            #[cfg(feature = "runtime-benchmarks")]
            type MaxKeys = frame_support::traits::ConstU32<1_000>;
            type MaxPeerInHeartbeats = MaxPeerInHeartbeats;
            type NextSessionRotation = Babe;
            type ReportUnresponsiveness = Offences;
            type RuntimeEvent = RuntimeEvent;
            type UnsignedPriority = ImOnlineUnsignedPriority;
            type ValidatorSet = Historical;
            type WeightInfo = weights::pallet_im_online::WeightInfo<Runtime>;
        }
        impl pallet_offences::Config for Runtime {
            type IdentificationTuple = pallet_session::historical::IdentificationTuple<Self>;
            type OnOffenceHandler = AuthorityMembers;
            type RuntimeEvent = RuntimeEvent;
        }
        impl pallet_session::Config for Runtime {
            type Keys = opaque::SessionKeys;
            type NextSessionRotation = Babe;
            type RuntimeEvent = RuntimeEvent;
            type SessionHandler = <opaque::SessionKeys as OpaqueKeys>::KeyTypeIdProviders;
            type SessionManager =
                pallet_session::historical::NoteHistoricalRoot<Self, AuthorityMembers>;
            type ShouldEndSession = Babe;
            type ValidatorId = AccountId;
            type ValidatorIdOf = sp_runtime::traits::ConvertInto;
            type WeightInfo = weights::pallet_session::WeightInfo<Runtime>;
        }
        impl pallet_session::historical::Config for Runtime {
            type FullIdentification = ValidatorFullIdentification;
            type FullIdentificationOf = FullIdentificationOfImpl;
        }
        impl pallet_grandpa::Config for Runtime {
            type EquivocationReportSystem = pallet_grandpa::EquivocationReportSystem<
                Self,
                Offences,
                Historical,
                ReportLongevity,
            >;
            type KeyOwnerProof = <Historical as KeyOwnerProofSystem<(KeyTypeId, GrandpaId)>>::Proof;
            type MaxAuthorities = MaxAuthorities;
            type MaxNominators = frame_support::traits::ConstU32<64>;
            type MaxSetIdSessionEntries = MaxSetIdSessionEntries;
            type RuntimeEvent = RuntimeEvent;
            type WeightInfo = weights::pallet_grandpa::WeightInfo<Runtime>;
        }
        parameter_types! {
            // BondingDuration::get() * SessionsPerEra::get();
            pub const MaxSetIdSessionEntries: u32 = 1000;
        }

        // ONCHAIN GOVERNANCE //

        #[cfg(feature = "runtime-benchmarks")]
        parameter_types! {
        pub const WorstCaseOrigin: WorstOrigin = WorstOrigin::Members(2, 3);
        }

        impl pallet_upgrade_origin::Config for Runtime {
            type Call = RuntimeCall;
            type RuntimeEvent = RuntimeEvent;
            type UpgradableOrigin = pallet_collective::EnsureProportionAtLeast<
                AccountId,
                TechnicalCommitteeInstance,
                2,
                3,
            >;
            type WeightInfo = weights::pallet_upgrade_origin::WeightInfo<Runtime>;
            #[cfg(feature = "runtime-benchmarks")]
            type WorstCaseOrigin = WorstCaseOrigin;
            #[cfg(feature = "runtime-benchmarks")]
            type WorstCaseOriginType = RawOrigin<AccountId, TechnicalCommitteeInstance>;
        }

        parameter_types! {
            pub const PreimageMaxSize: u32 = 4096 * 1024;
            pub const PreimageBaseDeposit: Balance = deposit(2, 64);
            pub const PreimageByteDeposit: Balance = deposit(0, 1);
        }

        impl pallet_preimage::Config for Runtime {
            type Consideration = ();
            type Currency = Balances;
            type ManagerOrigin = EnsureRoot<AccountId>;
            type RuntimeEvent = RuntimeEvent;
            type WeightInfo = weights::pallet_preimage::WeightInfo<Runtime>;
        }

        // UTILITIES //

        impl pallet_atomic_swap::Config for Runtime {
            type ProofLimit = frame_support::traits::ConstU32<1_024>;
            type RuntimeEvent = RuntimeEvent;
            type SwapAction = pallet_atomic_swap::BalanceSwapAction<AccountId, Balances>;
        }

        impl pallet_provide_randomness::Config for Runtime {
            type Currency = Balances;
            type GetCurrentEpochIndex = GetCurrentEpochIndex<Self>;
            type MaxRequests = frame_support::traits::ConstU32<100>;
            type OnFilledRandomness = ();
            type OnUnbalanced = HandleFees<TreasuryAccount, Balances>;
            type ParentBlockRandomness = pallet_babe::ParentBlockRandomness<Self>;
            type RandomnessFromOneEpochAgo = pallet_babe::RandomnessFromOneEpochAgo<Self>;
            type RequestPrice = frame_support::traits::ConstU64<2_000>;
            type RuntimeEvent = RuntimeEvent;
            type WeightInfo = weights::pallet_provide_randomness::WeightInfo<Runtime>;
        }

        parameter_types! {
            // One storage item; key size 32, value size 8; .
            pub const ProxyDepositBase: Balance = deposit(1, 8);
            // Additional storage item size of 33 bytes.
            pub const ProxyDepositFactor: Balance = deposit(0, 33);
            pub const AnnouncementDepositBase: Balance = deposit(1, 8);
            pub const AnnouncementDepositFactor: Balance = deposit(0, 66);
        }
        impl pallet_proxy::Config for Runtime {
            type AnnouncementDepositBase = AnnouncementDepositBase;
            type AnnouncementDepositFactor = AnnouncementDepositFactor;
            type CallHasher = BlakeTwo256;
            type Currency = Balances;
            type MaxPending = frame_support::traits::ConstU32<32>;
            type MaxProxies = frame_support::traits::ConstU32<32>;
            type ProxyDepositBase = ProxyDepositBase;
            type ProxyDepositFactor = ProxyDepositFactor;
            type ProxyType = ProxyType;
            type RuntimeCall = RuntimeCall;
            type RuntimeEvent = RuntimeEvent;
            type WeightInfo = weights::pallet_proxy::WeightInfo<Runtime>;
        }

        parameter_types! {
            pub const DepositBase: Balance = DEPOSIT_PER_ITEM;
            pub const DepositFactor: Balance = DEPOSIT_PER_BYTE * 32;
        }
        impl pallet_multisig::Config for Runtime {
            type Currency = Balances;
            type DepositBase = DepositBase;
            type DepositFactor = DepositFactor;
            type MaxSignatories = MaxSignatories;
            type RuntimeCall = RuntimeCall;
            type RuntimeEvent = RuntimeEvent;
            type WeightInfo = weights::pallet_multisig::WeightInfo<Runtime>;
        }

        impl pallet_utility::Config for Runtime {
            type PalletsOrigin = OriginCaller;
            type RuntimeCall = RuntimeCall;
            type RuntimeEvent = RuntimeEvent;
            type WeightInfo = weights::pallet_utility::WeightInfo<Runtime>;
        }

        parameter_types! {
            pub const Burn: Permill = Permill::zero();
            pub const ProposalBond: Permill = Permill::from_percent(1);
            pub const ProposalBondMaximum: Option<Balance> = None;
            pub const SpendPeriod: BlockNumber = DAYS;
            // Treasury account address:
            // gdev/gtest: 5EYCAe5ijiYfyeZ2JJCGq56LmPyNRAKzpG4QkoQkkQNB5e6Z
            pub const TreasuryPalletId: PalletId = PalletId(*b"py/trsry");
        }
        impl pallet_treasury::Config for Runtime {
            type AssetKind = ();
            type BalanceConverter = frame_support::traits::tokens::UnityAssetBalanceConversion;
            #[cfg(feature = "runtime-benchmarks")]
            type BenchmarkHelper = ();
            type Beneficiary = AccountId;
            type BeneficiaryLookup = AccountIdLookup<AccountId, ()>;
            type BlockNumberProvider = System;
            type Burn = Burn;
            type BurnDestination = ();
            type Currency = Balances;
            type MaxApprovals = frame_support::traits::ConstU32<100>;
            type PalletId = TreasuryPalletId;
            type Paymaster =
                frame_support::traits::tokens::pay::PayFromAccount<Balances, TreasuryAccount>;
            type PayoutPeriod = sp_core::ConstU32<10>;
            type RejectOrigin = TreasuryRejectOrigin;
            type RuntimeEvent = RuntimeEvent;
            type SpendFunds = TreasurySpendFunds<Self>;
            type SpendOrigin = frame_support::traits::NeverEnsureOrigin<Balance>;
            type SpendPeriod = SpendPeriod;
            type WeightInfo = weights::pallet_treasury::WeightInfo<Runtime>;
        }

        // UNIVERSAL DIVIDEND //

        impl pallet_universal_dividend::Config for Runtime {
            type Currency = Balances;
            #[cfg(feature = "runtime-benchmarks")]
            type IdtyAttr = Identity;
            type MaxPastReeval = frame_support::traits::ConstU32<160>;
            type MembersCount = common_runtime::providers::MembersCount<Membership>;
            type MembersStorage = common_runtime::providers::UdMembersStorage<Runtime>;
            type MomentIntoBalance = sp_runtime::traits::ConvertInto;
            type RuntimeEvent = RuntimeEvent;
            type SquareMoneyGrowthRate = SquareMoneyGrowthRate;
            type UdCreationPeriod = UdCreationPeriod;
            type UdReevalPeriod = UdReevalPeriod;
            type WeightInfo = weights::pallet_universal_dividend::WeightInfo<Runtime>;
        }

        // WEB OF TRUST //

        impl pallet_duniter_wot::Config for Runtime {
            type FirstIssuableOn = WotFirstCertIssuableOn;
            type MinCertForCreateIdtyRight = WotMinCertForCreateIdtyRight;
            type MinCertForMembership = WotMinCertForMembership;
        }

        parameter_types! {
            pub const ValidationPeriod: BlockNumber = 2 * MONTHS;
        }
        impl pallet_identity::Config for Runtime {
            type AccountLinker = Account;
            type AutorevocationPeriod = AutorevocationPeriod;
            type ChangeOwnerKeyPeriod = ChangeOwnerKeyPeriod;
            type CheckAccountWorthiness = Account;
            type CheckIdtyCallAllowed = Wot;
            type ConfirmPeriod = ConfirmPeriod;
            type DeletionPeriod = DeletionPeriod;
            type IdtyCreationPeriod = IdtyCreationPeriod;
            type IdtyData = IdtyData;
            type IdtyIndex = IdtyIndex;
            type IdtyNameValidator = IdtyNameValidatorImpl;
            type OnNewIdty = OnNewIdtyHandler<Runtime>;
            type OnRemoveIdty = OnRemoveIdtyHandler<Runtime>;
            type OwnerKeyChangePermission = OwnerKeyChangePermissionHandler<Runtime>;
            type RuntimeEvent = RuntimeEvent;
            type Signature = Signature;
            type Signer = <Signature as sp_runtime::traits::Verify>::Signer;
            type ValidationPeriod = ValidationPeriod;
            type WeightInfo = weights::pallet_identity::WeightInfo<Runtime>;
        }

        impl pallet_sudo::Config for Runtime {
            type RuntimeCall = RuntimeCall;
            type RuntimeEvent = RuntimeEvent;
            type WeightInfo = weights::pallet_sudo::WeightInfo<Runtime>;
        }

        impl pallet_membership::Config for Runtime {
            #[cfg(feature = "runtime-benchmarks")]
            type BenchmarkSetupHandler = common_runtime::providers::BenchmarkSetupHandler<Runtime>;
            type CheckMembershipOpAllowed = Wot;
            type IdtyAttr = Identity;
            type IdtyId = IdtyIndex;
            type MembershipPeriod = MembershipPeriod;
            type MembershipRenewalPeriod = MembershipRenewalPeriod;
            type OnNewMembership = OnNewMembershipHandler<Runtime>;
            type OnRemoveMembership = OnRemoveMembershipHandler<Runtime>;
            type RuntimeEvent = RuntimeEvent;
            type WeightInfo = weights::pallet_membership::WeightInfo<Runtime>;
        }

        impl pallet_certification::Config for Runtime {
            type CertPeriod = CertPeriod;
            type CheckCertAllowed = Wot;
            type IdtyAttr = Identity;
            type IdtyIndex = IdtyIndex;
            type MaxByIssuer = MaxByIssuer;
            type MinReceivedCertToBeAbleToIssueCert = MinReceivedCertToBeAbleToIssueCert;
            type OnNewcert = Wot;
            type OnRemovedCert = Wot;
            type RuntimeEvent = RuntimeEvent;
            type ValidityPeriod = ValidityPeriod;
            type WeightInfo = weights::pallet_certification::WeightInfo<Runtime>;
        }

        impl pallet_distance::Config for Runtime {
            type CheckRequestDistanceEvaluation = Wot;
            type Currency = Balances;
            type EvaluationPeriod = frame_support::traits::ConstU32<7>;
            type EvaluationPrice = frame_support::traits::ConstU64<1000>;
            type MaxRefereeDistance = MaxRefereeDistance;
            type MinAccessibleReferees = MinAccessibleReferees;
            type OnUnbalanced = HandleFees<TreasuryAccount, Balances>;
            type OnValidDistanceStatus = Wot;
            type RuntimeEvent = RuntimeEvent;
            type RuntimeHoldReason = RuntimeHoldReason;
            type WeightInfo = weights::pallet_distance::WeightInfo<Runtime>;
        }

        // SMITH-MEMBERS
        impl pallet_smith_members::Config for Runtime {
            type IdtyAttr = Identity;
            type IdtyIdOfAuthorityId = sp_runtime::traits::ConvertInto;
            type IdtyIndex = IdtyIndex;
            type IsWoTMember = common_runtime::providers::IsWoTMemberProvider<Runtime>;
            type MaxByIssuer = SmithMaxByIssuer;
            type MemberId = IdtyIndex;
            type MinCertForMembership = SmithWotMinCertForMembership;
            type OnSmithDelete = OnSmithDeletedHandler<Runtime>;
            type RuntimeEvent = RuntimeEvent;
            type SmithInactivityMaxDuration = SmithInactivityMaxDuration;
            type WeightInfo = weights::pallet_smith_members::WeightInfo<Runtime>;
        }

        pub struct TechnicalCommitteeDefaultVote;
        impl pallet_collective::DefaultVote for TechnicalCommitteeDefaultVote {
            fn default_vote(
                _prime_vote: Option<bool>,
                _yes_votes: u32,
                _no_votes: u32,
                _len: u32,
            ) -> bool {
                false
            }
        }
        parameter_types! {
            pub const TechnicalCommitteeMotionDuration: BlockNumber = 7 * DAYS;
            pub MaxWeight: Weight = Perbill::from_percent(50) * BlockWeights::get().max_block;
        }
        impl pallet_collective::Config<Instance2> for Runtime {
            type Consideration = ();
            #[cfg(not(feature = "runtime-benchmarks"))]
            type DefaultVote = TechnicalCommitteeDefaultVote;
            #[cfg(feature = "runtime-benchmarks")]
            type DefaultVote = pallet_collective::PrimeDefaultVote;
            type DisapproveOrigin = EnsureRoot<Self::AccountId>;
            type KillOrigin = EnsureRoot<Self::AccountId>;
            type MaxMembers = frame_support::pallet_prelude::ConstU32<100>;
            type MaxProposalWeight = MaxWeight;
            type MaxProposals = frame_support::pallet_prelude::ConstU32<20>;
            type MotionDuration = TechnicalCommitteeMotionDuration;
            type Proposal = RuntimeCall;
            type RuntimeEvent = RuntimeEvent;
            type RuntimeOrigin = RuntimeOrigin;
            type SetMembersOrigin = EnsureRoot<AccountId>;
            type WeightInfo = weights::pallet_collective::WeightInfo<Runtime>;
        }
    };
}
